<?php

namespace App\Http\Controllers;

use App\Models\Pos;
use App\Models\Brand;
use App\Models\Product;
use App\Models\SellTransaction;
use App\Models\Sale;
use App\Models\Customer;
use App\Models\MobileModel;
use App\Models\Exchange;
use App\Models\CustomerPayment;
use App\Models\Payment;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PosController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $brands = [];
        return view('pos.index', compact(['brands']));
    }

    public function fetchProfit(Request $request)
    {
        if ($request->action == 'fetchProfit') {
            // Get today's date
            $today = now()->format('Y-m-d');

            // Fetch profit from sell_transactions
            $profit1 = DB::table('sell_transactions')
                ->whereDate('date_time', $today)
                ->sum('profit');
            $profit1 = $profit1 ?: 0;

            // Fetch profit from direct_sale_report
            $profit2 = DB::table('direct_sales')
                ->whereDate('direct_date_time', $today)
                ->sum('direct_profit');
            $profit2 = $profit2 ?: 0;

            // Total profit
            $total_profit = $profit1 + $profit2;

            // Return response as JSON
            return response()->json([
                'success' => true,
                'total_profit' => $total_profit,
                'profit1' => $profit1,
                'profit2' => $profit2,
                'date' => $today
            ]);
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        Log::info('PosController@store called with data:', [
            'items' => $request->items,
            'method' => $request->method(),
            'all_data' => $request->all()
        ]);

        // Validate the request
        $request->validate([
            'customer_id' => 'nullable|exists:customers,id',
            'items' => 'nullable|json',
            'payments' => 'nullable|json',
            'payment_method' => 'required_without:payments|in:cash,credit,multiple',
            'pay_due' => 'nullable|numeric|min:0',
        ]);
        
        $tenant = app('tenant');

        // Get customer ID from the request
        $customerId = $request->customer_id === 'walk_in' ? null : $request->customer_id;
        
        // Get customer object if customerId is provided
        $customer = $customerId ? Customer::find($customerId) : null;

        $items = json_decode($request->items ?? '[]', true);
        $exchanges = json_decode($request->exchanges ?? '[]', true);
        $payments = json_decode($request->payments ?? '[]', true);
        $payDue = $request->pay_due ?? 0;

        // Validate and sanitize items data
        if (is_array($items)) {
            $items = array_filter($items, function($item) {
                return isset($item['product_id']) && 
                       isset($item['sold_quantity']) && 
                       $item['sold_quantity'] > 0;
            });
            
            // Ensure each item has required fields with defaults
            $items = array_map(function($item) {
                $item['price'] = isset($item['price']) && is_numeric($item['price']) 
                    ? (float)$item['price'] 
                    : 0;
                $item['sold_quantity'] = (int)($item['sold_quantity'] ?? 1);
                return $item;
            }, $items);
        } else {
            $items = [];
        }

        if (empty($items) && $payDue <= 0) {
            if ($request->ajax()) {
                return response()->json(['message' => 'No items selected and no payment towards due.'], 422);
            }
            return redirect()->back()->with('error', 'No items selected and no payment towards due.');
        }

        if (empty($items) && $request->payment_method !== 'cash') {
            if ($request->ajax()) {
                return response()->json(['message' => 'Payment towards due can only be made with cash.'], 422);
            }
            return redirect()->back()->with('error', 'Payment towards due can only be made with cash.');
        }

        $saleCreated = false;
        if (!empty($items)) {
            DB::beginTransaction();

            try {
                // Get POS pricing settings
                $settings = Setting::where('tenant_id', $tenant->id)->first();
                
                // Calculate total amount using buy_price from products with POS pricing
                $totalAmount = 0;
                foreach ($items as $item) {
                    try {
                        // Debug: Log the entire item structure
                        Log::info('Raw item data:', $item);
                        
                        // Get the product to determine base price
                        $product = Product::find($item['product_id']);
                        $basePrice = isset($item['price']) && is_numeric($item['price']) && $item['price'] > 0 
                            ? $item['price'] 
                            : ($product ? $product->buy_price : 0);
                        
                        // For sale calculations, use base price only (no POS markup)
                        $finalPrice = $basePrice;
                        
                        $soldQuantity = isset($item['sold_quantity']) ? $item['sold_quantity'] : 0;
                        $totalAmount += $finalPrice * $soldQuantity;
                        
                        // Log individual item processing
                        Log::info('Processing item for sale calculation:', [
                            'product_id' => $item['product_id'] ?? 'unknown',
                            'base_price' => $basePrice,
                            'final_price' => $finalPrice,
                            'sold_quantity' => $soldQuantity
                        ]);
                    } catch (\Exception $e) {
                        Log::error('Error processing item:', [
                            'item' => $item,
                            'error' => $e->getMessage()
                        ]);
                        throw $e;
                    }
                }

                // Calculate exchange discount
                $exchangeDiscount = 0;
                foreach ($exchanges as $exchange) {
                    $exchangeDiscount += $exchange['price'];
                }

                $netAmount = $totalAmount - $exchangeDiscount;

                // Create Sale record first
                $paymentMethod = !empty($payments) ? 'multiple' : ($request->payment_method ?? 'cash');
                $totalPaid = 0;
                if (!empty($payments)) {
                    $totalPaid = array_sum(array_column($payments, 'amount'));
                } elseif ($paymentMethod === 'cash') {
                    $totalPaid = $netAmount;
                }
                $paymentStatus = $totalPaid >= $netAmount ? 'paid' : ($totalPaid > 0 ? 'partial' : 'unpaid');
                $paidAmount = $totalPaid;
                $dueAmount = $netAmount - $totalPaid;

                $sale = Sale::create([
                    'customer_id' => $customer ? $customer->id : null,
                    'total_amount' => $totalAmount,
                    'discount' => $exchangeDiscount,
                    'net_amount' => $netAmount,
                    'paid_amount' => $paidAmount,
                    'due_amount' => $dueAmount,
                    'payment_method' => $paymentMethod,
                    'payment_status' => $paymentStatus,
                    'tenant_id' => app('tenant')->id ?? null,
                ]);

                foreach ($items as $item) {
                    try {
                        Log::info('Processing sale item:', $item);
                        
                        $product = Product::findOrFail($item['product_id']);

                        if ($product->stock < $item['sold_quantity']) {
                            DB::rollBack();
                            if ($request->ajax()) {
                                return response()->json(['message' => 'Insufficient stock for product: ' . $product->name], 422);
                            }
                            return redirect()->back()->with('error', 'Insufficient stock for product: ' . $product->name);
                        }

                        // Calculate base price from item or product
                        $basePrice = isset($item['price']) && is_numeric($item['price']) && $item['price'] > 0 
                            ? $item['price'] 
                            : $product->buy_price;
                        
                        // Use base price only for actual sale (no POS markup in transactions)
                        $finalPrice = $basePrice;
                        
                        // Calculate profit based on difference between selling price and buy price
                        $profit = ($finalPrice - $product->buy_price) * $item['sold_quantity'];
                    } catch (\Exception $e) {
                        Log::error('Error in sale item processing:', [
                            'item' => $item,
                            'error' => $e->getMessage()
                        ]);
                        DB::rollBack();
                        if ($request->ajax()) {
                            return response()->json(['message' => 'Error processing item: ' . $e->getMessage()], 500);
                        }
                        return redirect()->back()->with('error', 'Error processing item: ' . $e->getMessage());
                    }

                    SellTransaction::create([
                        'sale_id' => $sale->id,
                        'product_id' => $product->id,
                        'payment_method' => $request->payment_method,
                        'sell_price' => $finalPrice,  // Store actual selling price (base price, no POS markup)
                        'sold_quantity' => $item['sold_quantity'],
                        'profit' => $profit,
                        'warranty' => 'N/A', // Since not provided in POS
                        'date_time' => now(),
                        'type' => $product->type ?? 'Unknown',
                        'name' => $product->name,
                        'tenant_id' => app('tenant')->id,
                    ]);

                    $product->decrement('stock', $item['sold_quantity']);
                }
                // return $request;
                // Save exchanges linked to sale and add exchanged products to product table
                foreach ($exchanges as $exchange) {
                    $brandId = $exchange['brand_id'] ?: null;
                    $mobileModelId = $exchange['mobile_model_id'] ?: null;

                    // If custom brand provided, create it
                    if (!$brandId && !empty($exchange['brand_name'])) {
                        $brand = Brand::firstOrCreate(['name' => $exchange['brand_name']]);
                        $brandId = $brand->id;
                    }

                    // If custom model provided, create it
                    if (!$mobileModelId && !empty($exchange['model_name'])) {
                        $model = MobileModel::firstOrCreate([
                            'name' => $exchange['model_name'],
                            'brand_id' => $brandId
                        ]);
                        $mobileModelId = $model->id;
                    }

                    // Create the exchanged product in the products table
                    $productName = $exchange['brand_name'] . ' ' . $exchange['model_name'];
                    Product::create([
                        'name' => $productName,
                        'brand' => $brandId,
                        'mobile_model_id' => $mobileModelId,
                        'imei_or_sn' => $exchange['imei'],
                        'buy_price' => $exchange['price'],
                        'stock' => 1, // Assuming 1 unit for exchanged product
                        'condition' => 'old', // Exchanged products are typically used
                        'color' => 'Unknown', // Default color
                        'gb' => 0, // Default storage
                        'customer_info' => $customer ? $customer->id : null,
                        'tenant_id' => app('tenant')->id ?? null,
                        'type' => 'Exchanged Product',
                    ]);

                    Exchange::create([
                        'sale_id' => $sale->id,
                        'brand_id' => $brandId,
                        'mobile_model_id' => $mobileModelId,
                        'price' => $exchange['price'],
                        'imei' => $exchange['imei'],
                    ]);
                }

                // Create Payment records
                if (!empty($payments)) {
                    foreach ($payments as $payment) {
                        Payment::create([
                            'tenant_id' => app('tenant')->id,
                            'sale_id' => $sale->id,
                            'payment_type' => $payment['method'],
                            'amount' => $payment['amount'],
                        ]);
                    }
                } elseif ($paymentMethod === 'cash') {
                    Payment::create([
                        'tenant_id' => app('tenant')->id,
                        'sale_id' => $sale->id,
                        'payment_type' => 'cash',
                        'amount' => $paidAmount,
                    ]);
                }

                DB::commit();
                $saleCreated = true;
            } catch (\Exception $e) {
                DB::rollBack();
                Log::error('Error while storing sale: ' . $e->getMessage());
                if ($request->ajax()) {
                    return response()->json(['message' => 'An error occurred while processing the sale: ' . $e->getMessage()], 500);
                }
                return redirect()->back()->with('error', 'An error occurred while processing the sale: ' . $e->getMessage());
            }
        }

        // Handle payment towards previous due
        if ($customer && $payDue > 0) {
            $unpaidSales = Sale::where('customer_id', $customer->id)
                ->where('due_amount', '>', 0)
                ->get();

            $totalDue = $unpaidSales->sum('due_amount');

            if ($payDue <= $totalDue) {
                // Distribute sequentially from oldest
                $paidSaleIds = [];
                $remainingPay = $payDue;
                $unpaidSalesOrdered = Sale::where('customer_id', $customer->id)
                    ->where('due_amount', '>', 0)
                    ->orderBy('created_at', 'asc')
                    ->get();

                foreach ($unpaidSalesOrdered as $sale) {
                    if ($remainingPay <= 0) break;
                    $payAmount = min($remainingPay, $sale->due_amount);
                    $sale->paid_amount += $payAmount;
                    $sale->due_amount -= $payAmount;
                    if ($sale->due_amount <= 0) {
                        $sale->payment_status = 'paid';
                    } else {
                        $sale->payment_status = 'partial';
                    }
                    $sale->save();
                    $paidSaleIds[] = $sale->id;
                    $remainingPay -= $payAmount;
                }

                if (!empty($paidSaleIds)) {
                    CustomerPayment::create([
                        'customer_id' => $customer->id,
                        'type' => 'due_payment',
                        'amount' => $payDue - $remainingPay,
                        'sale_ids' => json_encode($paidSaleIds),
                        'notes' => 'Due payment for sales: ' . implode(', ', $paidSaleIds),
                        'tenant_id' => app('tenant')->id ?? null,
                    ]);
                }
            } else {
                // Pay full to all, advance the rest
                $paidSaleIds = [];
                $paidAmount = 0;
                foreach ($unpaidSales as $sale) {
                    $payAmount = $sale->due_amount;
                    $sale->paid_amount += $payAmount;
                    $sale->due_amount = 0;
                    $sale->payment_status = 'paid';
                    $sale->save();
                    $paidSaleIds[] = $sale->id;
                    $paidAmount += $payAmount;
                }

                if (!empty($paidSaleIds)) {
                    CustomerPayment::create([
                        'customer_id' => $customer->id,
                        'type' => 'due_payment',
                        'amount' => $paidAmount,
                        'sale_ids' => json_encode($paidSaleIds),
                        'notes' => 'Due payment for sales: ' . implode(', ', $paidSaleIds),
                        'tenant_id' => app('tenant')->id ?? null,
                    ]);
                }

                $advanceAmount = $payDue - $paidAmount;
                CustomerPayment::create([
                    'customer_id' => $customer->id,
                    'type' => 'advance',
                    'amount' => $advanceAmount,
                    'sale_ids' => null,
                    'notes' => 'Advance payment',
                    'tenant_id' => app('tenant')->id ?? null,
                ]);
            }
        }

        $message = $saleCreated ? 'Sale completed successfully!' : 'Payment towards due processed successfully!';

        if ($request->ajax()) {
            return response()->json(['message' => $message, 'sale_id' => $saleCreated ? $sale->id : null]);
        }

        return redirect()->back()->with('success', $message);
    }

    /**
     * Display the specified resource.
     */
    public function show(Pos $pos)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Pos $pos)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Pos $pos)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Pos $pos)
    {
        //
    }

    public function dashboard()
    {
        $tenantId = auth()->user()->tenant_id;
        
        $brands = Brand::all();
        $products = Product::where('tenant_id', $tenantId)->get();
        $mobileModels = MobileModel::all();
        $all_settings = Setting::where('tenant_id', $tenantId)->first();

        $customers = Customer::where('tenant_id', $tenantId)
            ->select('id', 'name', 'phone')
            ->get()
            ->map(function ($customer) {
                // Calculate total due amount for each customer
                $totalDueFromSales = Sale::where('customer_id', $customer->id)
                    ->where('return_status', '!=', 'returned')
                    ->sum('due_amount');
                    
                $advances = CustomerPayment::where('customer_id', $customer->id)
                    ->where('type', 'advance')
                    ->sum('amount');
                
                $netDue = $totalDueFromSales - $advances;
                $customer->total_due = max(0, $netDue);
                
                return $customer;
            });

        $walkInCustomer = Customer::where('tenant_id', $tenantId)
            ->where('name', 'Walk-in Customer')
            ->first();

        return view('pos.dashboard', compact('brands', 'products', 'customers', 'mobileModels', 'walkInCustomer', 'all_settings'));
    }

    public function getCustomerDue(Request $request)
    {
        $customerId = $request->customer_id;
        $totalDueFromSales = Sale::where('customer_id', $customerId)
            ->sum('due_amount');
            
        $duePayments = CustomerPayment::where('customer_id', $customerId)
            ->where('type', 'due_payment')
            ->sum('amount');

        $advances = CustomerPayment::where('customer_id', $customerId)
            ->where('type', 'advance')
            ->sum('amount');

        $netDue = $totalDueFromSales - $advances;
        $outstandingDue = max(0, $netDue);
        $advance = max(0, -$netDue);

        return response()->json(['due' => $outstandingDue, 'advance' => $advance]);
    }

    public function getModelsByBrand(Request $request)
    {
        $brandId = $request->brand_id;
        $models = MobileModel::where('brand_id', $brandId)->get(['id', 'name']);
        return response()->json($models);
    }
}
